// 默认设置
const DEFAULT_SETTINGS = {
    frequencyLevel: '3000',
    highlightColor: 'inherit',
    wordLookup: true,
    paragraphSpeech: true,
    aiAnalysis: true,
    showToolbar: true,
    highlight: true  // 添加高亮开关
};

// 获取设置
export async function getSettings() {
    try {
        const data = await chrome.storage.local.get('settings');
        // 合并默认设置和存储的设置，确保新字段有默认值
        return { ...DEFAULT_SETTINGS, ...data.settings };
    } catch (error) {
        console.error('加载设置失败:', error);
        return DEFAULT_SETTINGS;
    }
}

// 保存设置
export async function saveSettings(settings) {
    try {
        await chrome.storage.local.set({ settings });
        return { success: true };
    } catch (error) {
        console.error('保存设置失败:', error);
        return { 
            success: false, 
            error: error.message || '保存设置失败'
        };
    }
}

// 获取单个设置项
export async function getSetting(key) {
    const settings = await getSettings();
    return settings[key];
}

// 更新单个设置项
export async function updateSetting(key, value) {
    try {
        const settings = await getSettings();
        settings[key] = value;
        await saveSettings(settings);
        return { success: true };
    } catch (error) {
        console.error('更新设置失败:', error);
        return { 
            success: false, 
            error: error.message || '更新设置失败'
        };
    }
}

// 重置设置为默认值
export async function resetSettings() {
    try {
        await saveSettings(DEFAULT_SETTINGS);
        return { success: true };
    } catch (error) {
        console.error('重置设置失败:', error);
        return { 
            success: false, 
            error: error.message || '重置设置失败'
        };
    }
}

// 验证设置是否有效
export function validateSettings(settings) {
    // 验证必需的字段
    const requiredFields = [
        'frequencyLevel',
        'highlightColor',
        'wordLookup',
        'paragraphSpeech',
        'aiAnalysis',
        'showToolbar',
        'highlight'  // 添加新字段到验证列表
    ];

    for (const field of requiredFields) {
        if (!(field in settings)) {
            return {
                valid: false,
                error: `缺少必需的设置项: ${field}`
            };
        }
    }

    // 验证词频范围
    const validFrequencyLevels = ['lexicon','500', '1500', '3000', '5000', '8000'];
    if (!validFrequencyLevels.includes(settings.frequencyLevel)) {
        return {
            valid: false,
            error: `无效的词频范围: ${settings.frequencyLevel}`
        };
    }

    return { valid: true };
}
